<?php
/**
 * Server Status Dashboard
 * Styled to match StockMate/ItemMate design system
 */

$historyFile = __DIR__ . '/status_history.json';
$maxHistoryPoints = 8640; // 24h at 10s intervals

function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    return round($bytes / (1024 ** $pow), $precision) . ' ' . $units[$pow];
}

function formatKB($kb) { return formatBytes($kb * 1024); }



function getSystemData($lightweight = false) {
    $hostname = gethostname();
    $uptime = trim(shell_exec('uptime -p'));
    $kernel = php_uname('r');
    $os = php_uname('s') . ' ' . php_uname('m');
    $load = sys_getloadavg();

    $cpuInfo = shell_exec('cat /proc/cpuinfo | grep "model name" | head -1');
    $cpuModel = $cpuInfo ? trim(str_replace('model name', '', str_replace(':', '', $cpuInfo))) : 'Unknown';
    $cpuCores = intval(shell_exec('nproc'));

    $cpuStat1 = file('/proc/stat');
    usleep(100000);
    $cpuStat2 = file('/proc/stat');

    $cpu1 = explode(' ', preg_replace('/\s+/', ' ', $cpuStat1[0]));
    $cpu2 = explode(' ', preg_replace('/\s+/', ' ', $cpuStat2[0]));
    $idle1 = $cpu1[4]; $idle2 = $cpu2[4];
    $total1 = array_sum(array_slice($cpu1, 1, 7));
    $total2 = array_sum(array_slice($cpu2, 1, 7));
    $cpuUsage = max(0, min(100, 100 - (($idle2 - $idle1) / ($total2 - $total1) * 100)));

    $coreUsage = [];
    for ($i = 1; $i <= $cpuCores; $i++) {
        $core1 = explode(' ', preg_replace('/\s+/', ' ', $cpuStat1[$i]));
        $core2 = explode(' ', preg_replace('/\s+/', ' ', $cpuStat2[$i]));
        $diff = array_sum(array_slice($core2, 1, 7)) - array_sum(array_slice($core1, 1, 7));
        $usage = $diff > 0 ? 100 - (($core2[4] - $core1[4]) / $diff * 100) : 0;
        $coreUsage[] = round(max(0, min(100, $usage)), 1);
    }

    $memInfo = file('/proc/meminfo');
    $memData = [];
    foreach ($memInfo as $line) {
        $parts = preg_split('/\s+/', $line);
        $memData[rtrim($parts[0], ':')] = intval($parts[1]);
    }
    $memTotal = $memData['MemTotal'] ?? 0;
    $memAvailable = $memData['MemAvailable'] ?? ($memData['MemFree'] ?? 0);
    $memUsed = $memTotal - $memAvailable;
    $memPercent = $memTotal > 0 ? ($memUsed / $memTotal) * 100 : 0;
    $swapTotal = $memData['SwapTotal'] ?? 0;
    $swapFree = $memData['SwapFree'] ?? 0;
    $swapUsed = $swapTotal - $swapFree;
    $swapPercent = $swapTotal > 0 ? ($swapUsed / $swapTotal) * 100 : 0;

    $diskTotal = disk_total_space('/');
    $diskFree = disk_free_space('/');
    $diskUsed = $diskTotal - $diskFree;
    $diskPercent = $diskTotal > 0 ? ($diskUsed / $diskTotal) * 100 : 0;

    $netRx = 0; $netTx = 0;
    $netDev = @file('/proc/net/dev');
    if ($netDev) {
        foreach ($netDev as $line) {
            if (preg_match('/^\s*(eth|eno|ens|enp|wlan)\S*:\s*(\d+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\d+)/', $line, $m)) {
                $netRx += $m[2]; $netTx += $m[3];
            }
        }
    }

    // Network rate - compare with previous measurement (with file locking)
    $netFile = '/tmp/dashboard_net_prev.json';
    $netRxRate = 0; $netTxRate = 0;
    if (file_exists($netFile)) {
        $prev = @json_decode(@file_get_contents($netFile), true);
        if ($prev) {
            $elapsed = microtime(true) - ($prev['time'] ?? 0);
            if ($elapsed > 0 && $elapsed < 120) {
                $netRxRate = max(0, ($netRx - ($prev['rx'] ?? 0)) / $elapsed);
                $netTxRate = max(0, ($netTx - ($prev['tx'] ?? 0)) / $elapsed);
            }
        }
    }
    @file_put_contents($netFile, json_encode(['time' => microtime(true), 'rx' => $netRx, 'tx' => $netTx]), LOCK_EX);

    // Network connections - detailed TCP states
    $netConns = intval(trim(shell_exec("/usr/sbin/ss -tun state established 2>/dev/null | tail -n +2 | wc -l")));
    $tcpTimeWait = intval(trim(shell_exec("/usr/sbin/ss -tan state time-wait 2>/dev/null | tail -n +2 | wc -l")));
    $tcpCloseWait = intval(trim(shell_exec("/usr/sbin/ss -tan state close-wait 2>/dev/null | tail -n +2 | wc -l")));
    $tcpSynRecv = intval(trim(shell_exec("/usr/sbin/ss -tan state syn-recv 2>/dev/null | tail -n +2 | wc -l")));
    $tcpListen = intval(trim(shell_exec("/usr/sbin/ss -tln 2>/dev/null | tail -n +2 | wc -l")));
    $tcpTotal = intval(trim(shell_exec("/usr/sbin/ss -tan 2>/dev/null | tail -n +2 | wc -l")));

    // Open file descriptors (system-wide)
    $fileNr = @file_get_contents('/proc/sys/fs/file-nr');
    $openFds = 0; $maxFds = 0;
    if ($fileNr) {
        $parts = preg_split('/\s+/', trim($fileNr));
        $openFds = intval($parts[0] ?? 0);
        $maxFds = intval($parts[2] ?? 0);
    }

    // PHP-FPM pool status via process counting
    $fpmActive = intval(trim(shell_exec("ps aux 2>/dev/null | grep 'php-fpm.*pool' | grep -v grep | wc -l")));
    $fpmMaxChildren = intval(trim(shell_exec("grep -m1 'pm.max_children' /etc/php-fpm.d/*.conf 2>/dev/null | grep -oP '\\d+'"))) ?: 24;

    // Nginx worker connections
    $nginxWorkers = intval(trim(shell_exec("ps aux 2>/dev/null | grep 'nginx: worker' | grep -v grep | wc -l")));
    $nginxConns = intval(trim(shell_exec("/usr/sbin/ss -tan 2>/dev/null | grep ':80\\b' | grep ESTAB | wc -l")));
    $nginxConns443 = intval(trim(shell_exec("/usr/sbin/ss -tan 2>/dev/null | grep ':443\\b' | grep ESTAB | wc -l")));
    $nginxConns += $nginxConns443;

    // Unique IPs connected
    $uniqueIps = intval(trim(shell_exec("/usr/sbin/ss -tan state established 2>/dev/null | awk '{print \$4}' | grep -oP '[0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+' | sort -u | wc -l")));

    // Disk I/O from /proc/diskstats
    $diskReadBytes = 0; $diskWriteBytes = 0;
    $diskStats = @file('/proc/diskstats');
    if ($diskStats) {
        foreach ($diskStats as $line) {
            $cols = preg_split('/\s+/', trim($line));
            // Only count main devices (sda, vda, nvme0n1 etc), not partitions
            if (isset($cols[2]) && preg_match('/^(sd[a-z]|vd[a-z]|nvme\d+n\d+)$/', $cols[2])) {
                $diskReadBytes += (intval($cols[5] ?? 0)) * 512;  // sectors read * 512
                $diskWriteBytes += (intval($cols[9] ?? 0)) * 512; // sectors written * 512
            }
        }
    }

    // Disk I/O rate - compare with previous measurement stored in temp file (with file locking)
    $ioFile = '/tmp/dashboard_diskio_prev.json';
    $diskReadRate = 0; $diskWriteRate = 0;
    if (file_exists($ioFile)) {
        $prev = @json_decode(@file_get_contents($ioFile), true);
        if ($prev) {
            $elapsed = microtime(true) - ($prev['time'] ?? 0);
            if ($elapsed > 0 && $elapsed < 120) {
                $diskReadRate = max(0, ($diskReadBytes - ($prev['read'] ?? 0)) / $elapsed);
                $diskWriteRate = max(0, ($diskWriteBytes - ($prev['write'] ?? 0)) / $elapsed);
            }
        }
    }
    @file_put_contents($ioFile, json_encode(['time' => microtime(true), 'read' => $diskReadBytes, 'write' => $diskWriteBytes]), LOCK_EX);

    $processes = shell_exec('ps aux --sort=-%cpu | head -16');
    $processLines = explode("\n", trim($processes));
    $processList = [];
    foreach ($processLines as $i => $line) {
        if ($i === 0 || empty(trim($line))) continue;
        $cols = preg_split('/\s+/', $line, 11);
        $processList[] = [
            'user' => $cols[0] ?? '', 'pid' => $cols[1] ?? '',
            'cpu' => $cols[2] ?? '', 'mem' => $cols[3] ?? '',
            'command' => substr($cols[10] ?? '', 0, 60)
        ];
    }

    // Fast metrics result (always collected)
    $result = [
        'timestamp' => date('Y-m-d H:i:s'),
        'hostname' => $hostname, 'os' => $os, 'kernel' => $kernel, 'uptime' => $uptime,
        'load' => [round($load[0], 2), round($load[1], 2), round($load[2], 2)],
        'cpuModel' => $cpuModel, 'cpuCores' => $cpuCores,
        'cpuUsage' => round($cpuUsage, 1), 'coreUsage' => $coreUsage,
        'memUsed' => formatKB($memUsed), 'memTotal' => formatKB($memTotal),
        'memPercent' => round($memPercent, 1),
        'memBuffers' => formatKB($memData['Buffers'] ?? 0),
        'memCached' => formatKB($memData['Cached'] ?? 0),
        'swapUsed' => formatKB($swapUsed), 'swapTotal' => formatKB($swapTotal),
        'swapPercent' => round($swapPercent, 1),
        'diskUsed' => formatBytes($diskUsed), 'diskTotal' => formatBytes($diskTotal),
        'diskPercent' => round($diskPercent, 1),
        'netRx' => $netRx, 'netTx' => $netTx,
        'netRxRate' => round($netRxRate), 'netTxRate' => round($netTxRate),
        'netConns' => $netConns,
        'tcpTimeWait' => $tcpTimeWait, 'tcpCloseWait' => $tcpCloseWait,
        'tcpSynRecv' => $tcpSynRecv, 'tcpListen' => $tcpListen, 'tcpTotal' => $tcpTotal,
        'openFds' => $openFds, 'maxFds' => $maxFds,
        'fpmActive' => $fpmActive, 'fpmMaxChildren' => $fpmMaxChildren,
        'nginxWorkers' => $nginxWorkers, 'nginxConns' => $nginxConns,
        'uniqueIps' => $uniqueIps,
        'diskReadRate' => round($diskReadRate), 'diskWriteRate' => round($diskWriteRate),
        'processes' => $processList,
        'rxBytes' => $netRx,
        'txBytes' => $netTx,
    ];

    // Skip heavy operations in lightweight mode (AJAX polling)
    if ($lightweight) {
        return $result;
    }

    // Heavy operations: version info, public IP (only on full page load)
    return $result + [
        'phpVersion' => phpversion(),
        'serverSoftware' => $_SERVER['SERVER_SOFTWARE'] ?? 'CLI',
        'arch' => php_uname('m'),
        'fullKernel' => php_uname('r'),
        'bootTime' => trim(shell_exec("who -b 2>/dev/null | awk '{print $3, $4}'")),
        'timezone' => date_default_timezone_get(),
        'networkInterfaces' => trim(shell_exec("ip -br addr show 2>/dev/null | head -6")),
        'dnsServers' => trim(shell_exec("grep '^nameserver' /etc/resolv.conf 2>/dev/null | awk '{print $2}' | head -3 | tr '\n' ', '")),
        'selinux' => trim(shell_exec("getenforce 2>/dev/null") ?: 'N/A'),
        'mysqlVersion' => trim(shell_exec("mysql --version 2>/dev/null | awk '{print $3, $4, $5}'")),
        'nginxVersion' => trim(shell_exec("nginx -v 2>&1 | sed 's/.*nginx\\///' | head -1")),
        'phpFpmVersion' => trim(shell_exec("php-fpm -v 2>/dev/null | head -1 | awk '{print $2}'")),
        'composerVersion' => trim(shell_exec("composer --version 2>/dev/null | awk '{print $3}'")),
        'nodeVersion' => trim(shell_exec("node -v 2>/dev/null")),
        'npmVersion' => trim(shell_exec("npm -v 2>/dev/null")),
        'gitVersion' => trim(shell_exec("git --version 2>/dev/null | awk '{print $3}'")),
        'pythonVersion' => trim(shell_exec("python3 --version 2>/dev/null | awk '{print $2}'")),
        'openSSLVersion' => trim(shell_exec("openssl version 2>/dev/null | awk '{print $2}'")),
        'firewallZones' => trim(shell_exec("firewall-cmd --get-active-zones 2>/dev/null | head -4")),
        'defaultGateway' => trim(shell_exec("ip route 2>/dev/null | awk '/default/{print $3}'")),
        'publicIp' => trim(@file_get_contents('https://api.ipify.org') ?: ''),
        'listenPorts' => trim(shell_exec("/usr/sbin/ss -tlnp 2>/dev/null | awk 'NR>1{print $4}' | sed 's/.*://' | sort -un | tr '\n' ', '")),
        'defaultIface' => trim(shell_exec("ip route 2>/dev/null | awk '/default/{print $5}'")),
        'macAddress' => trim(shell_exec("ip link show 2>/dev/null | awk '/ether/{print $2; exit}'")),
        'cpuFreq' => trim(shell_exec("grep 'cpu MHz' /proc/cpuinfo 2>/dev/null | head -1 | awk -F: '{printf \"%.0f MHz\", $2}'")),
        'cpuCache' => trim(shell_exec("grep 'cache size' /proc/cpuinfo 2>/dev/null | head -1 | awk -F: '{print $2}'")),
        'cpuGovernor' => trim(shell_exec("cat /sys/devices/system/cpu/cpu0/cpufreq/scaling_governor 2>/dev/null") ?: 'N/A'),
        'memFree' => formatKB($memData['MemFree'] ?? 0),
        'memAvailable' => formatKB($memAvailable),
        'memBuffersRaw' => formatKB($memData['Buffers'] ?? 0),
        'memCachedRaw' => formatKB($memData['Cached'] ?? 0),
        'memShmem' => formatKB($memData['Shmem'] ?? 0),
        'memSlab' => formatKB($memData['Slab'] ?? 0),
        'diskInodes' => trim(shell_exec("df -i / 2>/dev/null | awk 'NR==2{print $3\"/\"$2\" (\"$5\")\"}'")) ?: 'N/A',
        'blockDevices' => trim(shell_exec("lsblk -dno NAME,SIZE,TYPE 2>/dev/null | head -4")),
        'virtType' => trim(shell_exec("systemd-detect-virt 2>/dev/null") ?: 'bare-metal'),
    ];
}

function storeHistory($data, $historyFile, $maxPoints) {
    $history = [];
    $lockFile = $historyFile . '.lock';
    $fp = fopen($lockFile, 'c');
    if (!$fp || !flock($fp, LOCK_EX)) {
        if ($fp) fclose($fp);
        return $history;
    }
    try {
        if (file_exists($historyFile)) {
            $history = json_decode(file_get_contents($historyFile), true) ?: [];
        }
        $history[] = [
            't' => $data['timestamp'],
            'cpu' => $data['cpuUsage'], 'mem' => $data['memPercent'],
            'swap' => $data['swapPercent'], 'load' => $data['load'][0],
            'netRx' => $data['netRx'], 'netTx' => $data['netTx'],
            'netRxRate' => $data['netRxRate'], 'netTxRate' => $data['netTxRate'],
            'conns' => $data['netConns'], 'tw' => $data['tcpTimeWait'],
            'fpm' => $data['fpmActive'], 'ips' => $data['uniqueIps'], 'fds' => $data['openFds'],
            'diskRead' => $data['diskReadRate'], 'diskWrite' => $data['diskWriteRate'],
        ];
        if (count($history) > $maxPoints) {
            $history = array_slice($history, -$maxPoints);
        }
        file_put_contents($historyFile, json_encode($history), LOCK_EX);
    } finally {
        flock($fp, LOCK_UN);
        fclose($fp);
    }
    return $history;
}

if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    $data = getSystemData(true);
    storeHistory($data, $historyFile, $maxHistoryPoints);
    // Only send the new data point, not the full history (JS accumulates client-side)
    echo json_encode($data);
    exit;
}

if (isset($_GET['download'])) {
    $zip = new ZipArchive();
    $tmpFile = tempnam(sys_get_temp_dir(), 'dashboard_') . '.zip';
    if ($zip->open($tmpFile, ZipArchive::CREATE) === true) {
        $baseDir = __DIR__;

        $zip->addFile($baseDir . '/index.php', 'index.php');

        $files = ['favicon.ico', 'favicon.svg'];
        foreach ($files as $f) {
            $path = $baseDir . '/' . $f;
            if (file_exists($path)) $zip->addFile($path, $f);
        }

        $zip->close();
        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="server-dashboard.zip"');
        header('Content-Length: ' . filesize($tmpFile));
        readfile($tmpFile);
        unlink($tmpFile);
        exit;
    }
}

$data = getSystemData();
$history = storeHistory($data, $historyFile, $maxHistoryPoints);
?>
<!DOCTYPE html>
<html lang="da">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Server Status - Dashboard</title>
    <link rel="icon" href="/favicon.ico" sizes="any">
    <link rel="icon" type="image/svg+xml" href="/favicon.svg">
    <link href="https://fonts.googleapis.com/css2?family=DM+Sans:wght@400;500;600;700&family=JetBrains+Mono:wght@400;500&display=swap" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        :root {
            --bg-primary: #f5f5f5; --bg-secondary: #ffffff; --bg-tertiary: #fafafa;
            --bg-hover: #f0f0f0; --text-primary: #111111; --text-secondary: #555555;
            --text-muted: #999999; --accent-green: #198754; --accent-green-hover: #157347;
            --accent-red: #dc3545; --accent-orange: #fd7e14; --accent-blue: #0d6efd;
            --accent-cyan: #0dcaf0; --accent-purple: #6f42c1; --border-color: #e0e0e0;
            --radius: 8px; --radius-sm: 6px;
        }
        body.dark {
            --bg-primary: #1a1a2e; --bg-secondary: #16213e; --bg-tertiary: #1e2a45;
            --bg-hover: #1e2a45; --text-primary: #e0e0e0; --text-secondary: #aaaaaa;
            --text-muted: #777777; --border-color: #2a2a4a;
        }
        body.midnight {
            --bg-primary: #0f0f1a; --bg-secondary: #12121f; --bg-tertiary: #181828;
            --bg-hover: #1a1a2c; --text-primary: #c8c8d4; --text-secondary: #8888a0;
            --text-muted: #555570; --border-color: #22223a;
            --accent-green: #22c55e; --accent-red: #ef4444; --accent-orange: #f59e0b;
            --accent-blue: #3b82f6; --accent-cyan: #06b6d4; --accent-purple: #8b5cf6;
        }
        body.abyss {
            --bg-primary: #050508; --bg-secondary: #0a0a10; --bg-tertiary: #0f0f18;
            --bg-hover: #111120; --text-primary: #b0b0c0; --text-secondary: #707088;
            --text-muted: #444458; --border-color: #161625;
            --accent-green: #10b981; --accent-red: #f43f5e; --accent-orange: #f59e0b;
            --accent-blue: #6366f1; --accent-cyan: #22d3ee; --accent-purple: #a855f7;
        }
        body.nord {
            --bg-primary: #2e3440; --bg-secondary: #3b4252; --bg-tertiary: #434c5e;
            --bg-hover: #4c566a; --text-primary: #eceff4; --text-secondary: #d8dee9;
            --text-muted: #7b88a1; --border-color: #4c566a;
            --accent-green: #a3be8c; --accent-red: #bf616a; --accent-orange: #d08770;
            --accent-blue: #81a1c1; --accent-cyan: #88c0d0; --accent-purple: #b48ead;
        }
        body.shadow {
            --bg-primary: #060609; --bg-secondary: #0b0b12; --bg-tertiary: #10101a;
            --bg-hover: #131322; --text-primary: #d0d0dd; --text-secondary: #9090a8;
            --text-muted: #606078; --border-color: #181830;
            --accent-green: #10b981; --accent-red: #f43f5e; --accent-orange: #f59e0b;
            --accent-blue: #6366f1; --accent-cyan: #22d3ee; --accent-purple: #a855f7;
        }
        body.dark, body.midnight, body.abyss, body.shadow, body.nord { color-scheme: dark; }
        body.dark .badge-secondary, body.midnight .badge-secondary, body.abyss .badge-secondary, body.shadow .badge-secondary, body.nord .badge-secondary { background: var(--bg-tertiary); color: var(--text-secondary); }
        body.dark th, body.midnight th, body.abyss th, body.shadow th, body.nord th { background: var(--bg-tertiary); color: var(--text-secondary); }
        body.dark .progress-bar-bg, body.midnight .progress-bar-bg, body.abyss .progress-bar-bg, body.shadow .progress-bar-bg, body.nord .progress-bar-bg { background: var(--bg-tertiary); }
        body.dark .core-card-bar, body.midnight .core-card-bar, body.abyss .core-card-bar, body.shadow .core-card-bar, body.nord .core-card-bar { background: var(--bg-tertiary); }

        /* Kompakt mode */
        body.kompakt { font-size: 12px; }
        body.kompakt .container { padding: 16px 20px; }
        body.kompakt .page-header { margin-bottom: 14px; }
        body.kompakt .page-title { font-size: 1.2rem; }
        body.kompakt .page-logo { width: 32px; height: 32px; font-size: 1rem; }
        body.kompakt .stats-grid { gap: 10px; margin-bottom: 16px; }
        body.kompakt .stat-card { padding: 12px 14px; }
        body.kompakt .stat-value { font-size: 1.3rem; }
        body.kompakt .stat-label { font-size: 0.75rem; }
        body.kompakt .stat-icon { font-size: 1.2rem; }
        body.kompakt .card { padding: 14px; margin-bottom: 14px; }
        body.kompakt .card-header { margin-bottom: 10px; }
        body.kompakt .card-title { font-size: 0.85rem; }
        body.kompakt .chart-container { height: 150px; }
        body.kompakt .core-grid { gap: 8px; }
        body.kompakt .core-card { padding: 8px 10px; gap: 4px; }
        body.kompakt .core-card-value { font-size: 0.9rem; }
        body.kompakt .core-card-label { font-size: 0.7rem; }
        body.kompakt th, body.kompakt td { padding: 6px 10px; }
        body.kompakt .game-card { padding: 14px; gap: 8px; }
        body.kompakt .progress-row { margin-bottom: 8px; }
        body.kompakt .sysinfo-grid { gap: 16px; }
        body.kompakt .sysinfo-row { font-size: 0.75rem; padding: 2px 0; }

        /* Stram mode */
        body.stram { font-size: 11.5px; line-height: 1.35; }
        body.stram .container { padding: 12px 16px; }
        body.stram .page-header { margin-bottom: 10px; }
        body.stram .page-header-left { gap: 10px; }
        body.stram .page-title { font-size: 1.1rem; }
        body.stram .page-description { font-size: 0.72rem; }
        body.stram .page-logo { width: 28px; height: 28px; font-size: 0.9rem; border-radius: 6px; }
        body.stram .page-meta { gap: 10px; font-size: 0.7rem; }
        body.stram .stats-grid { gap: 8px; margin-bottom: 12px; }
        body.stram .stat-card { padding: 10px 12px; }
        body.stram .stat-value { font-size: 1.1rem; }
        body.stram .stat-label { font-size: 0.7rem; }
        body.stram .stat-sub { font-size: 0.65rem; }
        body.stram .stat-icon { font-size: 1.1rem; }
        body.stram .card { padding: 10px 12px; margin-bottom: 10px; }
        body.stram .card-header { margin-bottom: 8px; }
        body.stram .card-title { font-size: 0.8rem; }
        body.stram .chart-container { height: 120px; }
        body.stram .grid-3 { gap: 12px; }
        body.stram .core-grid { gap: 6px; grid-template-columns: repeat(6, 1fr); }
        body.stram .core-card { padding: 6px 8px; gap: 3px; }
        body.stram .core-card-value { font-size: 0.8rem; }
        body.stram .core-card-label { font-size: 0.65rem; }
        body.stram .core-card-bar { height: 4px; }
        body.stram th, body.stram td { padding: 4px 8px; font-size: 0.8rem; }
        body.stram th { font-size: 0.7rem; }
        body.stram .game-card { padding: 10px 12px; gap: 6px; }
        body.stram .game-card-title { font-size: 0.9rem; }
        body.stram .game-meta-label { font-size: 0.65rem; }
        body.stram .game-meta-value { font-size: 0.8rem; }
        body.stram .game-card-meta { gap: 12px; }
        body.stram .progress-row { margin-bottom: 6px; }
        body.stram .progress-label { font-size: 0.8rem; margin-bottom: 2px; }
        body.stram .progress-bar-bg { height: 6px; }
        body.stram .progress-label-value { font-size: 0.75rem; }
        body.stram .time-range-btn { padding: 3px 10px; font-size: 0.7rem; }
        body.stram .time-range-btns { gap: 3px; }
        body.stram .badge { padding: 2px 8px; font-size: 0.7rem; }
        body.stram .sysinfo-grid { gap: 12px; }
        body.stram .sysinfo-row { font-size: 0.72rem; padding: 2px 0; }
        body.stram .sysinfo-group-title { font-size: 0.65rem; margin-bottom: 6px; }

        /* Minimal mode */
        body.minimal { font-size: 11px; line-height: 1.3; }
        body.minimal .container { padding: 8px 12px; }
        body.minimal .page-header { margin-bottom: 8px; }
        body.minimal .page-header-left { gap: 8px; }
        body.minimal .page-title { font-size: 1rem; }
        body.minimal .page-description { font-size: 0.7rem; }
        body.minimal .page-logo { width: 26px; height: 26px; font-size: 0.8rem; border-radius: 5px; }
        body.minimal .page-meta { gap: 8px; font-size: 0.65rem; }
        body.minimal .stats-grid { gap: 6px; margin-bottom: 8px; grid-template-columns: repeat(auto-fit, minmax(130px, 1fr)); }
        body.minimal .stat-card { padding: 8px 10px; }
        body.minimal .stat-card::before { width: 2px; }
        body.minimal .stat-value { font-size: 1rem; }
        body.minimal .stat-label { font-size: 0.65rem; margin-top: 0; }
        body.minimal .stat-sub { font-size: 0.6rem; margin-top: 1px; }
        body.minimal .stat-icon { font-size: 1rem; top: 8px; right: 8px; }
        body.minimal .card { padding: 8px 10px; margin-bottom: 8px; }
        body.minimal .card-header { margin-bottom: 6px; }
        body.minimal .card-title { font-size: 0.75rem; }
        body.minimal .card-title i { margin-right: 4px !important; }
        body.minimal .chart-container { height: 100px; }
        body.minimal .grid-2 { gap: 8px; }
        body.minimal .grid-3 { gap: 8px; }
        body.minimal .core-grid { gap: 4px; grid-template-columns: repeat(8, 1fr); }
        body.minimal .core-card { padding: 4px 6px; gap: 2px; }
        body.minimal .core-card-value { font-size: 0.7rem; }
        body.minimal .core-card-label { font-size: 0.6rem; }
        body.minimal .core-card-bar { height: 3px; }
        body.minimal th, body.minimal td { padding: 3px 6px; font-size: 0.75rem; }
        body.minimal th { font-size: 0.65rem; }
        body.minimal .game-card { padding: 8px 10px; gap: 6px; }
        body.minimal .game-card-title { font-size: 0.85rem; }
        body.minimal .game-meta-label { font-size: 0.6rem; }
        body.minimal .game-meta-value { font-size: 0.75rem; }
        body.minimal .game-card-link { font-size: 0.75rem; }
        body.minimal .game-card-meta { gap: 10px; }
        body.minimal .badge { padding: 2px 6px; font-size: 0.65rem; }
        body.minimal .sysinfo-grid { gap: 8px; }
        body.minimal .sysinfo-row { font-size: 0.68rem; padding: 1px 0; }
        body.minimal .sysinfo-group-title { font-size: 0.6rem; margin-bottom: 4px; }
        body.minimal .progress-row { margin-bottom: 4px; }
        body.minimal .progress-label { font-size: 0.75rem; margin-bottom: 2px; }
        body.minimal .progress-bar-bg { height: 5px; }
        body.minimal .progress-label-value { font-size: 0.7rem; }
        body.minimal .time-range-btn { padding: 2px 8px; font-size: 0.65rem; }
        body.minimal .time-range-btns { gap: 2px; }

        /* Settings panel */
        .settings-wrapper { position: relative; }
        .settings-btn {
            background: var(--bg-secondary); border: 1px solid var(--border-color);
            border-radius: var(--radius-sm); width: 34px; height: 34px;
            display: flex; align-items: center; justify-content: center;
            cursor: pointer; font-size: 1rem; color: var(--text-secondary);
            transition: border-color 0.2s, color 0.2s;
        }
        .settings-btn:hover { border-color: var(--accent-green); color: var(--accent-green); }
        .settings-panel {
            display: none; position: absolute; top: 42px; right: 0; z-index: 100;
            background: var(--bg-secondary); border: 1px solid var(--border-color);
            border-radius: var(--radius); padding: 16px; min-width: 260px;
            box-shadow: 0 8px 24px rgba(0,0,0,0.12);
        }
        .settings-panel.open { display: block; }
        .settings-group { margin-bottom: 14px; }
        .settings-group:last-child { margin-bottom: 0; }
        .settings-group-title {
            font-size: 0.7rem; font-weight: 700; text-transform: uppercase;
            letter-spacing: 0.05em; color: var(--text-muted); margin-bottom: 8px;
        }
        .settings-toggle {
            display: flex; align-items: center; justify-content: space-between;
            padding: 4px 0; cursor: pointer; font-size: 0.85rem; color: var(--text-primary);
        }
        .settings-toggle input[type="checkbox"] { accent-color: var(--accent-green); width: 16px; height: 16px; cursor: pointer; }
        .settings-section-item {
            display: flex; align-items: center; gap: 8px;
            padding: 3px 0; font-size: 0.8rem; color: var(--text-secondary); cursor: pointer;
        }
        .settings-section-item input[type="checkbox"] { accent-color: var(--accent-green); width: 14px; height: 14px; cursor: pointer; }
        .settings-section-item input[type="radio"] { accent-color: var(--accent-green); width: 14px; height: 14px; cursor: pointer; }
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'DM Sans', -apple-system, BlinkMacSystemFont, sans-serif;
            background: var(--bg-primary); color: var(--text-primary);
            line-height: 1.5; min-height: 100vh; font-size: 14px;
        }
        .container { max-width: 1400px; margin: 0 auto; padding: 28px 36px; }
        .page-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 24px; }
        .page-header-left { display: flex; align-items: center; gap: 14px; }
        .page-logo { width: 40px; height: 40px; background: var(--accent-green); border-radius: 8px; display: flex; align-items: center; justify-content: center; color: white; font-size: 1.2rem; }
        .page-title { font-size: 1.5rem; font-weight: 700; }
        .page-description { color: var(--text-secondary); font-size: 0.85rem; }
        .page-meta { display: flex; align-items: center; gap: 16px; font-family: 'JetBrains Mono', monospace; font-size: 0.75rem; color: var(--text-muted); }
        .live-dot { display: inline-block; width: 8px; height: 8px; background: var(--accent-green); border-radius: 50%; animation: pulse 2s infinite; }
        @keyframes pulse { 0%,100% { opacity: 1; } 50% { opacity: 0.4; } }

        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-bottom: 24px; }
        .stat-card { background: var(--bg-secondary); border: 1px solid var(--border-color); border-radius: var(--radius); padding: 20px; position: relative; overflow: hidden; }
        .stat-card::before { content: ''; position: absolute; top: 0; left: 0; width: 3px; height: 100%; }
        .stat-card.green::before { background: var(--accent-green); }
        .stat-card.blue::before { background: var(--accent-blue); }
        .stat-card.cyan::before { background: var(--accent-cyan); }
        .stat-card.orange::before { background: var(--accent-orange); }
        .stat-card.purple::before { background: var(--accent-purple); }
        .stat-icon { position: absolute; top: 16px; right: 16px; font-size: 1.5rem; opacity: 0.1; }
        .stat-value { font-size: 1.75rem; font-weight: 700; font-family: 'JetBrains Mono', monospace; }
        .stat-label { color: var(--text-secondary); font-size: 0.8rem; margin-top: 2px; }
        .stat-sub { color: var(--text-muted); font-size: 0.75rem; font-family: 'JetBrains Mono', monospace; margin-top: 4px; }

        .card { background: var(--bg-secondary); border: 1px solid var(--border-color); border-radius: var(--radius); padding: 20px; margin-bottom: 20px; min-width: 0; overflow: hidden; }
        .card-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 16px; flex-wrap: wrap; gap: 8px; }
        .card-title { font-size: 0.95rem; font-weight: 600; }

        .grid-2 { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; }
        .grid-3 { display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 20px; min-width: 0; }
        .grid-3 > * { min-width: 0; }

        .progress-row { margin-bottom: 12px; }
        .progress-label { display: flex; justify-content: space-between; margin-bottom: 4px; font-size: 0.85rem; }
        .progress-label-text { color: var(--text-secondary); font-weight: 500; }
        .progress-label-value { font-family: 'JetBrains Mono', monospace; font-weight: 600; font-size: 0.8rem; }
        .progress-bar-bg { height: 8px; background: var(--bg-primary); border-radius: 4px; overflow: hidden; }
        .progress-bar-fill { height: 100%; border-radius: 4px; transition: width 0.5s ease; }
        .progress-green { background: var(--accent-green); } .progress-blue { background: var(--accent-blue); }
        .progress-orange { background: var(--accent-orange); } .progress-red { background: var(--accent-red); }
        .progress-cyan { background: var(--accent-cyan); } .progress-purple { background: var(--accent-purple); }

        /* CPU Cores - full width, prominent */
        .core-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 12px; }
        .core-card {
            background: var(--bg-tertiary); border: 1px solid var(--border-color);
            border-radius: var(--radius-sm); padding: 12px 14px;
            display: flex; flex-direction: column; gap: 6px;
        }
        .core-card-header { display: flex; justify-content: space-between; align-items: center; }
        .core-card-label { font-family: 'JetBrains Mono', monospace; font-size: 0.8rem; font-weight: 600; color: var(--text-secondary); }
        .core-card-value { font-family: 'JetBrains Mono', monospace; font-size: 1.1rem; font-weight: 700; }
        .core-card-bar { height: 6px; background: var(--bg-primary); border-radius: 3px; overflow: hidden; }
        .core-card-fill { height: 100%; border-radius: 3px; transition: width 0.5s; }

        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 10px 14px; text-align: left; border-bottom: 1px solid var(--border-color); }
        th { font-weight: 600; color: var(--text-secondary); font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.04em; background: var(--bg-tertiary); }
        tbody tr:hover { background: var(--bg-hover); }
        td { font-size: 0.85rem; }
        .mono { font-family: 'JetBrains Mono', monospace; font-size: 0.85em; }

        .badge { display: inline-flex; align-items: center; padding: 3px 10px; border-radius: 4px; font-size: 0.75rem; font-weight: 600; }
        .badge-success { background: rgba(25,135,84,0.1); color: var(--accent-green); }
        .badge-danger { background: rgba(220,53,69,0.1); color: var(--accent-red); }
        .badge-warning { background: rgba(253,126,20,0.1); color: var(--accent-orange); }
        .badge-info { background: rgba(13,202,240,0.1); color: #0aa2c0; }
        .badge-secondary { background: #f0f0f0; color: var(--text-secondary); }

        /* Game server cards */
        .game-card { background: var(--bg-secondary); border: 1px solid var(--border-color); border-radius: var(--radius); padding: 20px; display: flex; flex-direction: column; gap: 12px; }
        .game-card-header { display: flex; justify-content: space-between; align-items: center; }
        .game-card-title { font-weight: 600; font-size: 1rem; display: flex; align-items: center; gap: 8px; }
        .game-card-meta { display: flex; gap: 16px; flex-wrap: wrap; }
        .game-meta-item { display: flex; flex-direction: column; font-size: 0.8rem; }
        .game-meta-label { color: var(--text-muted); font-size: 0.7rem; text-transform: uppercase; letter-spacing: 0.04em; }
        .game-meta-value { font-family: 'JetBrains Mono', monospace; font-weight: 600; }
        .game-card-link { display: inline-flex; align-items: center; gap: 6px; color: var(--accent-green); font-size: 0.85rem; font-weight: 500; text-decoration: none; }
        .game-card-link:hover { text-decoration: underline; }
        .sysinfo-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 24px; }
        .sysinfo-grid > * { min-width: 0; }
        .sysinfo-group-title { font-size: 0.7rem; font-weight: 700; text-transform: uppercase; letter-spacing: 0.06em; color: var(--accent-green); margin-bottom: 8px; }
        .sysinfo-row { display: flex; justify-content: space-between; gap: 12px; padding: 3px 0; border-bottom: 1px solid var(--border-color); font-size: 0.8rem; }
        .sysinfo-row:last-child { border-bottom: none; }
        .sysinfo-label { color: var(--text-muted); white-space: nowrap; flex-shrink: 0; }
        .sysinfo-value { text-align: right; word-break: break-all; color: var(--text-primary); }

        .chart-container { position: relative; height: 200px; min-width: 0; overflow: hidden; }
        .chart-footer { display: flex; align-items: center; justify-content: center; gap: 10px; margin-top: 4px; font-size: 0.7rem; font-weight: 600; color: var(--text-muted); letter-spacing: 0.02em; }
        .chart-footer .cf-dot { display: inline-block; width: 8px; height: 8px; border-radius: 50%; flex-shrink: 0; }
        .chart-footer .cf-item { display: flex; align-items: center; gap: 4px; font-family: 'JetBrains Mono', monospace; }

        /* Time range buttons */
        .time-range-btns { display: flex; gap: 4px; }
        .time-range-btn {
            padding: 4px 12px; border-radius: 4px; border: 1px solid var(--border-color);
            background: var(--bg-secondary); color: var(--text-secondary);
            font-family: 'JetBrains Mono', monospace; font-size: 0.75rem;
            cursor: pointer; transition: all 0.15s; font-weight: 500;
        }
        .time-range-btn:hover { border-color: var(--accent-green); color: var(--accent-green); }
        .time-range-btn.active { background: var(--accent-green); color: white; border-color: var(--accent-green); }
        .refresh-btn {
            padding: 4px 10px; border-radius: 4px; border: 1px solid var(--border-color);
            background: var(--bg-tertiary); color: var(--text-secondary); font-size: 0.75rem;
            font-family: 'JetBrains Mono', monospace; cursor: pointer; font-weight: 500;
        }
        .refresh-btn:hover { border-color: var(--accent-green); color: var(--accent-green); }
        .refresh-btn.active { background: var(--accent-green); color: white; border-color: var(--accent-green); }
        .download-dashboard-link {
            display: flex; align-items: center; gap: 6px; color: var(--accent-green);
            font-size: 0.8rem; font-weight: 600; text-decoration: none; padding: 6px 0;
            white-space: nowrap;
        }
        .download-dashboard-link:hover { text-decoration: underline; }

        @media (max-width: 1024px) {
            .container { padding: 16px; }
            .grid-2, .grid-3 { grid-template-columns: 1fr; }
            .stats-grid { grid-template-columns: repeat(2, 1fr); }
            .core-grid { grid-template-columns: repeat(2, 1fr); }
            .project-cards { grid-template-columns: 1fr; }
            .sysinfo-grid { grid-template-columns: repeat(2, 1fr); }
        }
        @media (max-width: 640px) {
            .stats-grid { grid-template-columns: 1fr; }
            .core-grid { grid-template-columns: 1fr; }
            .sysinfo-grid { grid-template-columns: 1fr; }
            .page-header { flex-direction: column; align-items: flex-start; gap: 12px; }
        }
    </style>
</head>
<body>
<div class="container">
    <div class="page-header">
        <div class="page-header-left">
            <div class="page-logo"><i class="bi bi-hdd-rack"></i></div>
            <div>
                <div class="page-title">Server Dashboard</div>
                <div class="page-description"><?= htmlspecialchars($data['hostname']) ?> &mdash; <?= htmlspecialchars($data['os']) ?></div>
            </div>
        </div>
        <div class="page-meta">
            <span><span class="live-dot"></span> Live</span>
            <span id="timestamp"><?= $data['timestamp'] ?></span>
            <span><?= htmlspecialchars($data['uptime']) ?></span>
            <div class="settings-wrapper">
                <button class="settings-btn" onclick="toggleSettingsPanel()" title="Indstillinger">⚙</button>
                <div class="settings-panel" id="settingsPanel">
                    <div class="settings-group">
                        <div class="settings-group-title">Tema</div>
                        <label class="settings-section-item"><input type="radio" name="themeMode" value="light" onchange="setTheme('light')"> Lys</label>
                        <label class="settings-section-item"><input type="radio" name="themeMode" value="dark" onchange="setTheme('dark')"> Mørk</label>
                        <label class="settings-section-item"><input type="radio" name="themeMode" value="midnight" onchange="setTheme('midnight')"> Midnight</label>
                        <label class="settings-section-item"><input type="radio" name="themeMode" value="abyss" onchange="setTheme('abyss')"> Abyss</label>
                        <label class="settings-section-item"><input type="radio" name="themeMode" value="shadow" onchange="setTheme('shadow')"> Shadow</label>
                        <label class="settings-section-item"><input type="radio" name="themeMode" value="nord" onchange="setTheme('nord')"> Nord</label>
                    </div>
                    <div class="settings-group">
                        <div class="settings-group-title">Layout</div>
                        <label class="settings-section-item"><input type="radio" name="layoutMode" value="normal" onchange="setLayout('normal')"> Komfort</label>
                        <label class="settings-section-item"><input type="radio" name="layoutMode" value="kompakt" onchange="setLayout('kompakt')"> Kompakt</label>
                        <label class="settings-section-item"><input type="radio" name="layoutMode" value="stram" onchange="setLayout('stram')"> Stram</label>
                        <label class="settings-section-item"><input type="radio" name="layoutMode" value="minimal" onchange="setLayout('minimal')"> Minimal</label>
                    </div>
                    <div class="settings-group">
                        <div class="settings-group-title">Opdateringsfrekvens</div>
                        <div style="display:flex; gap:4px; flex-wrap:wrap;">
                            <button class="refresh-btn" data-interval="2" onclick="setRefreshInterval(2)">2s</button>
                            <button class="refresh-btn" data-interval="5" onclick="setRefreshInterval(5)">5s</button>
                            <button class="refresh-btn active" data-interval="10" onclick="setRefreshInterval(10)">10s</button>
                            <button class="refresh-btn" data-interval="30" onclick="setRefreshInterval(30)">30s</button>
                            <button class="refresh-btn" data-interval="60" onclick="setRefreshInterval(60)">60s</button>
                        </div>
                    </div>
                    <div class="settings-group">
                        <div class="settings-group-title">Sektioner</div>
                        <label class="settings-section-item"><input type="checkbox" checked data-section-toggle="stats" onchange="toggleSection('stats', this.checked)">Stat Cards</label>
                        <label class="settings-section-item"><input type="checkbox" checked data-section-toggle="charts" onchange="toggleSection('charts', this.checked)">Statistik</label>
                        <label class="settings-section-item"><input type="checkbox" checked data-section-toggle="cores" onchange="toggleSection('cores', this.checked)">CPU Kerner</label>
                        <label class="settings-section-item"><input type="checkbox" checked data-section-toggle="sysinfo" onchange="toggleSection('sysinfo', this.checked)">System Info</label>
                        <label class="settings-section-item"><input type="checkbox" checked data-section-toggle="network" onchange="toggleSection('network', this.checked)">Netværk & Server</label>
                        <label class="settings-section-item"><input type="checkbox" checked data-section-toggle="processes" onchange="toggleSection('processes', this.checked)">Top Processer</label>
                    </div>
                    <div class="settings-group" style="border-top:1px solid var(--border-color); padding-top:10px;">
                        <a href="?download=1" class="download-dashboard-link"><i class="bi bi-download"></i> Download Dashboard (.zip)</a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Stat cards -->
    <div class="stats-grid" data-section="stats">
        <div class="stat-card green">
            <span class="stat-icon"><i class="bi bi-cpu"></i></span>
            <div class="stat-value" id="cpuValue"><?= $data['cpuUsage'] ?>%</div>
            <div class="stat-label">CPU Forbrug</div>
            <div class="stat-sub" id="cpuSub"><?= $data['cpuCores'] ?> kerner</div>
        </div>
        <div class="stat-card blue">
            <span class="stat-icon"><i class="bi bi-memory"></i></span>
            <div class="stat-value" id="memValue"><?= $data['memPercent'] ?>%</div>
            <div class="stat-label">RAM Forbrug</div>
            <div class="stat-sub" id="memSub"><?= $data['memUsed'] ?> / <?= $data['memTotal'] ?></div>
        </div>
        <div class="stat-card cyan">
            <span class="stat-icon"><i class="bi bi-hdd"></i></span>
            <div class="stat-value" id="diskValue"><?= $data['diskPercent'] ?>%</div>
            <div class="stat-label">Disk Forbrug</div>
            <div class="stat-sub" id="diskSub"><?= $data['diskUsed'] ?> / <?= $data['diskTotal'] ?></div>
        </div>
        <div class="stat-card orange">
            <span class="stat-icon"><i class="bi bi-speedometer"></i></span>
            <div class="stat-value" id="loadValue"><?= $data['load'][0] ?></div>
            <div class="stat-label">Load Average</div>
            <div class="stat-sub" id="loadSub"><?= implode(' / ', $data['load']) ?></div>
        </div>
        <div class="stat-card purple">
            <span class="stat-icon"><i class="bi bi-arrow-down-up"></i></span>
            <div class="stat-value" id="swapValue"><?= $data['swapPercent'] ?>%</div>
            <div class="stat-label">Swap Forbrug</div>
            <div class="stat-sub" id="swapSub"><?= $data['swapUsed'] ?> / <?= $data['swapTotal'] ?></div>
        </div>
        <div class="stat-card green">
            <span class="stat-icon"><i class="bi bi-hdd-network"></i></span>
            <div class="stat-value" id="netConnsValue"><?= $data['netConns'] ?></div>
            <div class="stat-label">Forbindelser</div>
            <div class="stat-sub" id="netRateSub">↓ <?= formatBytes($data['netRxRate']) ?>/s &nbsp; ↑ <?= formatBytes($data['netTxRate']) ?>/s</div>
        </div>
    </div>

    <!-- Charts with time range selector -->
    <div class="card" data-section="charts" style="margin-bottom:20px;">
        <div class="card-header">
            <div class="card-title"><i class="bi bi-graph-up" style="margin-right:6px; opacity:0.4;"></i>Statistik over tid</div>
            <div class="time-range-btns">
                <button class="time-range-btn active" data-range="30" onclick="setTimeRange(30)">5 min</button>
                <button class="time-range-btn" data-range="90" onclick="setTimeRange(90)">15 min</button>
                <button class="time-range-btn" data-range="180" onclick="setTimeRange(180)">30 min</button>
                <button class="time-range-btn" data-range="360" onclick="setTimeRange(360)">1 time</button>
                <button class="time-range-btn" data-range="2160" onclick="setTimeRange(2160)">6 timer</button>
                <button class="time-range-btn" data-range="8640" onclick="setTimeRange(8640)">24 timer</button>
                <button class="time-range-btn" data-range="0" onclick="setTimeRange(0)">Alt</button>
            </div>
        </div>
        <div class="grid-3" style="margin-bottom:12px;">
            <div>
                <div class="chart-container"><canvas id="cpuChart"></canvas></div>
                <div class="chart-footer">CPU (%)</div>
            </div>
            <div>
                <div class="chart-container"><canvas id="memChart"></canvas></div>
                <div class="chart-footer">Memory (%)</div>
            </div>
            <div>
                <div class="chart-container"><canvas id="diskIoChart"></canvas></div>
                <div class="chart-footer"><span class="cf-item"><span class="cf-dot" style="background:#0dcaf0"></span>Read</span> <span class="cf-item"><span class="cf-dot" style="background:#fd7e14"></span>Write</span> &mdash; Disk I/O</div>
            </div>
        </div>
        <div class="grid-3">
            <div>
                <div class="chart-container"><canvas id="networkChart"></canvas></div>
                <div class="chart-footer"><span class="cf-item"><span class="cf-dot" style="background:#198754"></span>RX</span> <span class="cf-item"><span class="cf-dot" style="background:#dc3545"></span>TX</span> &mdash; Netværk I/O</div>
            </div>
            <div>
                <div class="chart-container"><canvas id="loadChart"></canvas></div>
                <div class="chart-footer">Load Average</div>
            </div>
            <div>
                <div class="chart-container"><canvas id="swapChart"></canvas></div>
                <div class="chart-footer">Swap (%)</div>
            </div>
        </div>
    </div>


    <!-- CPU Cores -->
    <div class="card" data-section="cores" style="margin-bottom:20px;">
        <div class="card-header">
            <div class="card-title"><i class="bi bi-cpu" style="margin-right:6px; opacity:0.4;"></i>CPU Kerner</div>
            <div style="display:flex; align-items:center; gap:12px;">
                <span class="badge badge-secondary"><?= $data['cpuCores'] ?> kerner</span>
                <span class="badge badge-info"><?= htmlspecialchars($data['cpuModel']) ?></span>
            </div>
        </div>
        <div id="coreGrid" class="core-grid">
            <?php foreach ($data['coreUsage'] as $i => $usage):
                $color = $usage > 80 ? 'var(--accent-red)' : ($usage > 50 ? 'var(--accent-orange)' : 'var(--accent-green)');
            ?>
            <div class="core-card">
                <div class="core-card-header">
                    <span class="core-card-label">CPU <?= $i ?></span>
                    <span class="core-card-value" style="color:<?= $color ?>"><?= $usage ?>%</span>
                </div>
                <div class="core-card-bar">
                    <div class="core-card-fill" style="width:<?= $usage ?>%; background:<?= $color ?>;"></div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Top Processes -->
    <div class="card" data-section="processes" style="margin-bottom:20px;">
        <div class="card-header">
            <div class="card-title"><i class="bi bi-list-task" style="margin-right:6px; opacity:0.4;"></i>Top Processer (CPU)</div>
        </div>
        <div style="overflow-x:auto;">
            <table>
                <thead><tr><th>User</th><th>PID</th><th>%CPU</th><th>%MEM</th><th>Command</th></tr></thead>
                <tbody id="processTable">
                    <?php foreach ($data['processes'] as $p):
                        $cpuVal = floatval($p['cpu']);
                        $cpuColor = $cpuVal > 50 ? 'var(--accent-red)' : ($cpuVal > 20 ? 'var(--accent-orange)' : 'var(--text-primary)');
                    ?>
                    <tr>
                        <td class="mono"><?= htmlspecialchars($p['user']) ?></td>
                        <td class="mono"><?= htmlspecialchars($p['pid']) ?></td>
                        <td class="mono" style="color:<?= $cpuColor ?>; font-weight:600;"><?= htmlspecialchars($p['cpu']) ?></td>
                        <td class="mono"><?= htmlspecialchars($p['mem']) ?></td>
                        <td class="mono" style="max-width:400px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;"><?= htmlspecialchars($p['command']) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Network & Server Health -->
    <div class="grid-3" data-section="network" style="margin-bottom: 20px;">
        <div class="card">
            <div class="card-title" style="margin-bottom:16px;">TCP Forbindelser</div>
            <div style="font-size:0.85rem;">
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">Established</span><span class="mono" id="tcpEstab"><?= $data['netConns'] ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">TIME_WAIT</span><span class="mono" style="<?= $data['tcpTimeWait'] > 200 ? 'color:var(--accent-orange);' : '' ?>" id="tcpTw"><?= $data['tcpTimeWait'] ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">CLOSE_WAIT</span><span class="mono" style="<?= $data['tcpCloseWait'] > 20 ? 'color:var(--accent-red);' : '' ?>" id="tcpCw"><?= $data['tcpCloseWait'] ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">SYN_RECV</span><span class="mono" style="<?= $data['tcpSynRecv'] > 50 ? 'color:var(--accent-red);' : '' ?>" id="tcpSyn"><?= $data['tcpSynRecv'] ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0;"><span style="color:var(--text-muted);">LISTEN / Total</span><span class="mono"><span id="tcpListen"><?= $data['tcpListen'] ?></span> / <span id="tcpTotal"><?= $data['tcpTotal'] ?></span></span></div>
            </div>
        </div>
        <div class="card">
            <div class="card-title" style="margin-bottom:16px;">Nginx + PHP-FPM</div>
            <div style="font-size:0.85rem;">
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">Nginx Workers</span><span class="mono" id="nginxWorkers"><?= $data['nginxWorkers'] ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">HTTP Conns</span><span class="mono" id="nginxConns"><?= $data['nginxConns'] ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">PHP-FPM Workers</span><span class="mono" style="<?= $data['fpmActive'] >= $data['fpmMaxChildren'] ? 'color:var(--accent-red);' : '' ?>" id="fpmActive"><?= $data['fpmActive'] ?> / <?= $data['fpmMaxChildren'] ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0;"><span style="color:var(--text-muted);">Unikke IP'er</span><span class="mono" id="uniqueIps"><?= $data['uniqueIps'] ?></span></div>
            </div>
        </div>
        <div class="card">
            <div class="card-title" style="margin-bottom:16px;">Netværk I/O</div>
            <div style="font-size:0.85rem;">
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">Open FDs</span><span class="mono" id="openFds"><?= number_format($data['openFds']) ?></span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0; border-bottom:1px solid var(--border-color);"><span style="color:var(--text-muted);">RX</span><span class="mono" id="netRxVal"><?= formatBytes($data['netRxRate']) ?>/s</span></div>
                <div style="display:flex; justify-content:space-between; padding:4px 0;"><span style="color:var(--text-muted);">TX</span><span class="mono" id="netTxVal"><?= formatBytes($data['netTxRate']) ?>/s</span></div>
            </div>
        </div>
    </div>

    <!-- Server Stats Charts -->
    <div class="card" data-section="network" style="margin-bottom:20px;">
        <div class="card-header">
            <div class="card-title"><i class="bi bi-graph-up-arrow" style="margin-right:6px; opacity:0.4;"></i>Server Statistik over tid</div>
        </div>
        <div style="display:grid; grid-template-columns:repeat(4, 1fr); gap:20px; min-width:0;">
            <div>
                <div class="chart-container"><canvas id="connsChart"></canvas></div>
                <div class="chart-footer"><span class="cf-item"><span class="cf-dot" style="background:#198754"></span>Established</span> <span class="cf-item"><span class="cf-dot" style="background:#fd7e14"></span>TIME_WAIT</span> &mdash; TCP Forbindelser</div>
            </div>
            <div>
                <div class="chart-container"><canvas id="fpmChart"></canvas></div>
                <div class="chart-footer">PHP-FPM Workers</div>
            </div>
            <div>
                <div class="chart-container"><canvas id="ipsChart"></canvas></div>
                <div class="chart-footer">Unikke IP'er</div>
            </div>
            <div>
                <div class="chart-container"><canvas id="fdsChart"></canvas></div>
                <div class="chart-footer">Open File Descriptors</div>
            </div>
        </div>
    </div>


    <!-- System Information -->
    <div class="card" data-section="sysinfo" style="margin-bottom:0;">
        <div class="card-header">
            <div class="card-title"><i class="bi bi-info-circle" style="margin-right:6px; opacity:0.4;"></i>System Information</div>
        </div>
        <div class="sysinfo-grid">
            <div class="sysinfo-group">
                <div class="sysinfo-group-title">System</div>
                <div class="sysinfo-row"><span class="sysinfo-label">Hostname</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['hostname']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">OS</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['os']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Kernel</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['fullKernel']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Arkitektur</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['arch']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Virtualisering</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['virtType']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Boot tid</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['bootTime']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Uptime</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['uptime']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Load Avg</span><span class="sysinfo-value mono"><?= $data['load'][0] ?> / <?= $data['load'][1] ?> / <?= $data['load'][2] ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Tidszone</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['timezone']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">SELinux</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['selinux']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Open FDs</span><span class="sysinfo-value mono"><?= $data['openFds'] ?></span></div>
            </div>
            <div class="sysinfo-group">
                <div class="sysinfo-group-title">Netværk</div>
                <div class="sysinfo-row"><span class="sysinfo-label">Offentlig IP</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['publicIp'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Gateway</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['defaultGateway'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">DNS</span><span class="sysinfo-value mono"><?= htmlspecialchars(rtrim($data['dnsServers'], ', ') ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">MAC</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['macAddress'] ?: 'N/A') ?></span></div>
                <?php
                $interfaces = array_filter(explode("\n", $data['networkInterfaces']));
                foreach ($interfaces as $iface):
                    $parts = preg_split('/\s+/', trim($iface), 3);
                    $ifName = $parts[0] ?? '';
                    $ifState = $parts[1] ?? '';
                    $ifAddr = $parts[2] ?? '';
                ?>
                <div class="sysinfo-row"><span class="sysinfo-label"><?= htmlspecialchars($ifName) ?></span><span class="sysinfo-value mono"><?= htmlspecialchars($ifAddr) ?> <span style="color:<?= $ifState === 'UP' ? 'var(--accent-green)' : 'var(--text-muted)' ?>;">(<?= htmlspecialchars($ifState) ?>)</span></span></div>
                <?php endforeach; ?>
                <div class="sysinfo-row"><span class="sysinfo-label">Lytteporte</span><span class="sysinfo-value mono"><?= htmlspecialchars(rtrim($data['listenPorts'], ', ') ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Total RX</span><span class="sysinfo-value mono"><?= formatBytes($data['rxBytes']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Total TX</span><span class="sysinfo-value mono"><?= formatBytes($data['txBytes']) ?></span></div>
                <?php if ($data['firewallZones']): ?>
                <div class="sysinfo-row"><span class="sysinfo-label">Firewall</span><span class="sysinfo-value mono"><?= htmlspecialchars(str_replace("\n", ', ', $data['firewallZones'])) ?></span></div>
                <?php endif; ?>
            </div>
            <div class="sysinfo-group">
                <div class="sysinfo-group-title">Hardware</div>
                <div class="sysinfo-row"><span class="sysinfo-label">CPU</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['cpuModel']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Kerner</span><span class="sysinfo-value mono"><?= $data['cpuCores'] ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Frekvens</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['cpuFreq'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Cache</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['cpuCache'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Governor</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['cpuGovernor']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">RAM Total</span><span class="sysinfo-value mono"><?= $data['memTotal'] ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Swap</span><span class="sysinfo-value mono"><?= $data['swapUsed'] ?> / <?= $data['swapTotal'] ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Disk /</span><span class="sysinfo-value mono"><?= $data['diskUsed'] ?> / <?= $data['diskTotal'] ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Inodes</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['diskInodes']) ?></span></div>
                <?php foreach (array_filter(explode("\n", $data['blockDevices'])) as $blk):
                    $bp = preg_split('/\s+/', trim($blk));
                ?>
                <div class="sysinfo-row"><span class="sysinfo-label"><?= htmlspecialchars($bp[0] ?? '') ?></span><span class="sysinfo-value mono"><?= htmlspecialchars(($bp[1] ?? '') . ' ' . ($bp[2] ?? '')) ?></span></div>
                <?php endforeach; ?>
            </div>
            <div class="sysinfo-group">
                <div class="sysinfo-group-title">Software</div>
                <div class="sysinfo-row"><span class="sysinfo-label">Nginx</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['nginxVersion']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">PHP</span><span class="sysinfo-value mono"><?= $data['phpVersion'] ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">PHP-FPM</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['phpFpmVersion']) ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">MySQL</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['mysqlVersion'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Python</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['pythonVersion'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Node.js</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['nodeVersion'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">npm</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['npmVersion'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Git</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['gitVersion'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">Composer</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['composerVersion'] ?: 'N/A') ?></span></div>
                <div class="sysinfo-row"><span class="sysinfo-label">OpenSSL</span><span class="sysinfo-value mono"><?= htmlspecialchars($data['openSSLVersion'] ?: 'N/A') ?></span></div>
            </div>
        </div>
    </div>
</div>

<script>
// --- Preferences (apply before render to avoid flash) ---
const defaultPrefs = { theme: 'light', layout: 'normal', refreshInterval: 10, sections: { stats: true, charts: true, cores: true, sysinfo: true, network: true, processes: true } };
const allThemes = ['dark', 'midnight', 'abyss', 'shadow', 'nord'];
const darkThemes = ['dark', 'midnight', 'abyss', 'shadow', 'nord'];
function loadPrefs() {
    try {
        const s = localStorage.getItem('dashboardPrefs');
        if (!s) return { ...defaultPrefs, sections: { ...defaultPrefs.sections } };
        const p = JSON.parse(s);
        // Migrate old compact boolean
        if (typeof p.compact === 'boolean' && !p.layout) { p.layout = p.compact ? 'kompakt' : 'normal'; delete p.compact; }
        if (p.layout === 'compact') p.layout = 'kompakt';
        if (p.layout === 'crazy-compact') p.layout = 'minimal';
        if (typeof p.dark === 'boolean') { p.theme = p.dark ? 'dark' : 'light'; delete p.dark; }
        return { ...defaultPrefs, ...p, sections: { ...defaultPrefs.sections, ...(p.sections || {}) } };
    }
    catch(e) { return { ...defaultPrefs, sections: { ...defaultPrefs.sections } }; }
}
function savePrefs(p) { localStorage.setItem('dashboardPrefs', JSON.stringify(p)); }
let prefs = loadPrefs();
if (prefs.theme && prefs.theme !== 'light') document.body.classList.add(prefs.theme);
if (prefs.layout && prefs.layout !== 'normal') document.body.classList.add(prefs.layout);
Object.entries(prefs.sections).forEach(([k, v]) => { if (!v) { const el = document.querySelector('[data-section="'+k+'"]'); if (el) el.style.display = 'none'; } });

function toggleSettingsPanel() {
    document.getElementById('settingsPanel').classList.toggle('open');
}
document.addEventListener('click', function(e) {
    const wrapper = document.querySelector('.settings-wrapper');
    if (wrapper && !wrapper.contains(e.target)) {
        document.getElementById('settingsPanel').classList.remove('open');
    }
});
function setTheme(theme) {
    allThemes.forEach(t => document.body.classList.remove(t));
    if (theme !== 'light') document.body.classList.add(theme);
    prefs.theme = theme;
    savePrefs(prefs);
    updateChartColors();
}
function setLayout(mode) {
    document.body.classList.remove('kompakt', 'stram', 'minimal');
    if (mode !== 'normal') document.body.classList.add(mode);
    prefs.layout = mode;
    savePrefs(prefs);
}
function toggleSection(key, val) {
    prefs.sections[key] = val;
    const el = document.querySelector('[data-section="'+key+'"]');
    if (el) el.style.display = val ? '' : 'none';
    savePrefs(prefs);
}
// Sync checkboxes on load
document.addEventListener('DOMContentLoaded', function() {
    const themeRadio = document.querySelector('input[name="themeMode"][value="' + (prefs.theme || 'light') + '"]');
    if (themeRadio) themeRadio.checked = true;
    const layoutRadio = document.querySelector('input[name="layoutMode"][value="' + (prefs.layout || 'normal') + '"]');
    if (layoutRadio) layoutRadio.checked = true;
    Object.entries(prefs.sections).forEach(([k, v]) => {
        const cb = document.querySelector('[data-section-toggle="'+k+'"]');
        if (cb) cb.checked = v;
    });
    // Set active refresh button
    const ri = prefs.refreshInterval || 10;
    document.querySelectorAll('.refresh-btn').forEach(b => {
        b.classList.toggle('active', parseInt(b.dataset.interval) === ri);
    });
});

let allHistory = <?= json_encode($history) ?>;
let currentRange = 30;
let prevNetRx = null, prevNetTx = null;

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function getVisibleHistory() {
    if (currentRange === 0 || allHistory.length <= currentRange) return allHistory;
    return allHistory.slice(-currentRange);
}

function timeLabels(data) {
    return data.map(d => {
        const parts = d.t.split(' ')[1];
        return parts ? parts.substring(0, 5) : '';
    });
}

function isDark() { return darkThemes.includes(prefs.theme); }
function getGridColor() { return isDark() ? 'rgba(255,255,255,0.07)' : 'rgba(0,0,0,0.05)'; }
function getTickColor() { return isDark() ? '#777' : '#999'; }

function makeChartDefaults() {
    return {
        responsive: true, maintainAspectRatio: false,
        animation: { duration: 300 },
        plugins: { legend: { display: false } },
        scales: {
            x: { grid: { display: false }, ticks: { maxTicksLimit: 10, font: { family: "'JetBrains Mono'", size: 10 }, color: getTickColor() } },
            y: { grid: { color: getGridColor() }, ticks: { font: { family: "'JetBrains Mono'", size: 10 }, color: getTickColor() } }
        }
    };
}

let chartDefaults = makeChartDefaults();
const visible = getVisibleHistory();

const cpuChart = new Chart(document.getElementById('cpuChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [
            { label: 'CPU %', data: visible.map(d => d.cpu), borderColor: '#198754', backgroundColor: 'rgba(25,135,84,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }
        ]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, max: 100, ticks: { ...chartDefaults.scales.y.ticks, callback: v => v + '%' } } } }
});

const memChart = new Chart(document.getElementById('memChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [
            { label: 'Memory %', data: visible.map(d => d.mem), borderColor: '#0d6efd', backgroundColor: 'rgba(13,110,253,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }
        ]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, max: 100, ticks: { ...chartDefaults.scales.y.ticks, callback: v => v + '%' } } } }
});

function formatIoRate(bytes) {
    if (bytes >= 1073741824) return (bytes / 1073741824).toFixed(1) + ' GB/s';
    if (bytes >= 1048576) return (bytes / 1048576).toFixed(1) + ' MB/s';
    if (bytes >= 1024) return (bytes / 1024).toFixed(1) + ' KB/s';
    return bytes + ' B/s';
}

const diskIoChart = new Chart(document.getElementById('diskIoChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [
            { label: 'Read', data: visible.map(d => d.diskRead ?? 0), borderColor: '#0dcaf0', backgroundColor: 'rgba(13,202,240,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 },
            { label: 'Write', data: visible.map(d => d.diskWrite ?? 0), borderColor: '#fd7e14', backgroundColor: 'rgba(253,126,20,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }
        ]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, beginAtZero: true, ticks: { ...chartDefaults.scales.y.ticks, callback: v => formatIoRate(v) } } } }
});

const networkChart = new Chart(document.getElementById('networkChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [
            { label: 'RX', data: visible.map(d => d.netRxRate ?? 0), borderColor: '#198754', backgroundColor: 'rgba(25,135,84,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 },
            { label: 'TX', data: visible.map(d => d.netTxRate ?? 0), borderColor: '#dc3545', backgroundColor: 'rgba(220,53,69,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }
        ]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, beginAtZero: true, ticks: { ...chartDefaults.scales.y.ticks, callback: v => formatIoRate(v) } } } }
});

const loadChart = new Chart(document.getElementById('loadChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [{ label: 'Load (1m)', data: visible.map(d => d.load), borderColor: '#fd7e14', backgroundColor: 'rgba(253,126,20,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0 } } }
});

const swapChart = new Chart(document.getElementById('swapChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [
            { label: 'Swap %', data: visible.map(d => d.swap ?? 0), borderColor: '#6f42c1', backgroundColor: 'rgba(111,66,193,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }
        ]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, max: 100, ticks: { ...chartDefaults.scales.y.ticks, callback: v => v + '%' } } } }
});

const connsChart = new Chart(document.getElementById('connsChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [
            { label: 'Established', data: visible.map(d => d.conns ?? 0), borderColor: '#198754', backgroundColor: 'rgba(25,135,84,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 },
            { label: 'TIME_WAIT', data: visible.map(d => d.tw ?? 0), borderColor: '#fd7e14', backgroundColor: 'rgba(253,126,20,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }
        ]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, beginAtZero: true } } }
});

const fpmChart = new Chart(document.getElementById('fpmChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [{ label: 'PHP-FPM Workers', data: visible.map(d => d.fpm ?? 0), borderColor: '#6f42c1', backgroundColor: 'rgba(111,66,193,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, beginAtZero: true } } }
});

const ipsChart = new Chart(document.getElementById('ipsChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [{ label: 'Unikke IP\'er', data: visible.map(d => d.ips ?? 0), borderColor: '#0d6efd', backgroundColor: 'rgba(13,110,253,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, beginAtZero: true } } }
});

const fdsChart = new Chart(document.getElementById('fdsChart'), {
    type: 'line',
    data: {
        labels: timeLabels(visible),
        datasets: [{ label: 'Open FDs', data: visible.map(d => d.fds ?? 0), borderColor: '#dc3545', backgroundColor: 'rgba(220,53,69,0.1)', fill: true, tension: 0.3, pointRadius: 0, borderWidth: 2 }]
    },
    options: { ...chartDefaults, scales: { ...chartDefaults.scales, y: { ...chartDefaults.scales.y, min: 0, beginAtZero: true } } }
});

function updateChartColors() {
    const gridColor = getGridColor();
    const tickColor = getTickColor();
    [cpuChart, memChart, diskIoChart, networkChart, connsChart, loadChart, swapChart, fpmChart, ipsChart, fdsChart].forEach(chart => {
        chart.options.scales.y.grid.color = gridColor;
        chart.options.scales.y.ticks.color = tickColor;
        chart.options.scales.x.ticks.color = tickColor;
        chart.update('none');
    });
}

function setTimeRange(range) {
    currentRange = range;
    document.querySelectorAll('.time-range-btn').forEach(b => {
        b.classList.toggle('active', parseInt(b.dataset.range) === range);
    });
    updateCharts();
}

function updateCharts() {
    const v = getVisibleHistory();
    const labels = timeLabels(v);

    cpuChart.data.labels = labels;
    cpuChart.data.datasets[0].data = v.map(d => d.cpu);
    cpuChart.update('none');

    memChart.data.labels = labels;
    memChart.data.datasets[0].data = v.map(d => d.mem);
    memChart.update('none');

    diskIoChart.data.labels = labels;
    diskIoChart.data.datasets[0].data = v.map(d => d.diskRead ?? 0);
    diskIoChart.data.datasets[1].data = v.map(d => d.diskWrite ?? 0);
    diskIoChart.update('none');

    networkChart.data.labels = labels;
    networkChart.data.datasets[0].data = v.map(d => d.netRxRate ?? 0);
    networkChart.data.datasets[1].data = v.map(d => d.netTxRate ?? 0);
    networkChart.update('none');

    connsChart.data.labels = labels;
    connsChart.data.datasets[0].data = v.map(d => d.conns ?? 0);
    connsChart.data.datasets[1].data = v.map(d => d.tw ?? 0);
    connsChart.update('none');

    loadChart.data.labels = labels;
    loadChart.data.datasets[0].data = v.map(d => d.load);
    loadChart.update('none');

    swapChart.data.labels = labels;
    swapChart.data.datasets[0].data = v.map(d => d.swap ?? 0);
    swapChart.update('none');

    fpmChart.data.labels = labels;
    fpmChart.data.datasets[0].data = v.map(d => d.fpm ?? 0);
    fpmChart.update('none');

    ipsChart.data.labels = labels;
    ipsChart.data.datasets[0].data = v.map(d => d.ips ?? 0);
    ipsChart.update('none');

    fdsChart.data.labels = labels;
    fdsChart.data.datasets[0].data = v.map(d => d.fds ?? 0);
    fdsChart.update('none');
}

function updateDashboard() {
    fetch('?ajax=1')
        .then(r => r.json())
        .then(data => {
            document.getElementById('timestamp').textContent = data.timestamp;
            document.getElementById('cpuValue').textContent = data.cpuUsage + '%';
            document.getElementById('memValue').textContent = data.memPercent + '%';
            document.getElementById('memSub').textContent = data.memUsed + ' / ' + data.memTotal;
            document.getElementById('diskValue').textContent = data.diskPercent + '%';
            document.getElementById('diskSub').textContent = data.diskUsed + ' / ' + data.diskTotal;
            document.getElementById('loadValue').textContent = data.load[0];
            document.getElementById('loadSub').textContent = data.load.join(' / ');
            document.getElementById('swapValue').textContent = data.swapPercent + '%';
            document.getElementById('swapSub').textContent = data.swapUsed + ' / ' + data.swapTotal;

            document.getElementById('netConnsValue').textContent = data.netConns;
            document.getElementById('netRateSub').innerHTML = '↓ ' + formatIoRate(data.netRxRate) + ' &nbsp; ↑ ' + formatIoRate(data.netTxRate);

            // Network & Server section
            document.getElementById('tcpEstab').textContent = data.netConns;
            document.getElementById('tcpTw').textContent = data.tcpTimeWait;
            document.getElementById('tcpTw').style.color = data.tcpTimeWait > 200 ? 'var(--accent-orange)' : '';
            document.getElementById('tcpCw').textContent = data.tcpCloseWait;
            document.getElementById('tcpCw').style.color = data.tcpCloseWait > 20 ? 'var(--accent-red)' : '';
            document.getElementById('tcpSyn').textContent = data.tcpSynRecv;
            document.getElementById('tcpSyn').style.color = data.tcpSynRecv > 50 ? 'var(--accent-red)' : '';
            document.getElementById('tcpListen').textContent = data.tcpListen;
            document.getElementById('tcpTotal').textContent = data.tcpTotal;
            document.getElementById('nginxWorkers').textContent = data.nginxWorkers;
            document.getElementById('nginxConns').textContent = data.nginxConns;
            document.getElementById('fpmActive').textContent = data.fpmActive + ' / ' + data.fpmMaxChildren;
            document.getElementById('fpmActive').style.color = data.fpmActive >= data.fpmMaxChildren ? 'var(--accent-red)' : '';
            document.getElementById('uniqueIps').textContent = data.uniqueIps;
            document.getElementById('openFds').textContent = data.openFds.toLocaleString();
            document.getElementById('netRxVal').textContent = formatIoRate(data.netRxRate);
            document.getElementById('netTxVal').textContent = formatIoRate(data.netTxRate);


            // Cores
            const coreGrid = document.getElementById('coreGrid');
            coreGrid.innerHTML = data.coreUsage.map((usage, i) => {
                const color = usage > 80 ? 'var(--accent-red)' : usage > 50 ? 'var(--accent-orange)' : 'var(--accent-green)';
                return `<div class="core-card">
                    <div class="core-card-header">
                        <span class="core-card-label">CPU ${i}</span>
                        <span class="core-card-value" style="color:${color}">${usage}%</span>
                    </div>
                    <div class="core-card-bar"><div class="core-card-fill" style="width:${usage}%; background:${color};"></div></div>
                </div>`;
            }).join('');


            // Processes
            const tbody = document.getElementById('processTable');
            tbody.innerHTML = data.processes.map(p => {
                const cpuVal = parseFloat(p.cpu);
                const cpuColor = cpuVal > 50 ? 'var(--accent-red)' : cpuVal > 20 ? 'var(--accent-orange)' : 'var(--text-primary)';
                return `<tr>
                    <td class="mono">${escapeHtml(p.user)}</td>
                    <td class="mono">${escapeHtml(p.pid)}</td>
                    <td class="mono" style="color:${cpuColor}; font-weight:600;">${escapeHtml(p.cpu)}</td>
                    <td class="mono">${escapeHtml(p.mem)}</td>
                    <td class="mono" style="max-width:400px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap;">${escapeHtml(p.command)}</td>
                </tr>`;
            }).join('');

            // Accumulate history client-side from each AJAX data point
            allHistory.push({
                t: data.timestamp, cpu: data.cpuUsage, mem: data.memPercent,
                swap: data.swapPercent, load: data.load[0],
                netRx: data.netRx, netTx: data.netTx,
                netRxRate: data.netRxRate, netTxRate: data.netTxRate,
                conns: data.netConns, tw: data.tcpTimeWait,
                fpm: data.fpmActive, ips: data.uniqueIps, fds: data.openFds,
                diskRead: data.diskReadRate, diskWrite: data.diskWriteRate,
            });
            if (allHistory.length > 8640) allHistory = allHistory.slice(-8640);
            updateCharts();
        })
        .catch(err => console.error('Update failed:', err));
}

let refreshTimer = setInterval(updateDashboard, (prefs.refreshInterval || 10) * 1000);

function setRefreshInterval(seconds) {
    prefs.refreshInterval = seconds;
    savePrefs(prefs);
    clearInterval(refreshTimer);
    refreshTimer = setInterval(updateDashboard, seconds * 1000);
    document.querySelectorAll('.refresh-btn').forEach(b => {
        b.classList.toggle('active', parseInt(b.dataset.interval) === seconds);
    });
}
</script>
</body>
</html>
